package com.copote.integration;

import com.copote.integration.nodejs.NodeEngine;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.TestPropertySource;

import java.util.Map;

@SpringBootTest
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
@TestPropertySource("classpath:springtest.properties")
public class NodeJsEngineTest {

    @Autowired
    NodeEngine engine;

    @Test
    @Order(1)
    void EngineInit() {

        assert engine.get_nodePoolMaxTotal() > 0;
        assert engine.get_nodePoolMinIdle() >= 0;
        assert engine.get_nodePoolMaxIdle() > 0;

        assert engine.get_nodeContextPool().getNumIdle() <= engine.get_nodePoolMaxIdle();
        assert engine.get_nodeContextPool().getNumIdle() >= engine.get_nodePoolMinIdle();
        assert engine.get_nodeContextPool().getNumIdle() + engine.get_nodeContextPool().getNumActive() <= engine.get_nodePoolMaxTotal();

        assert engine.get_jsFunctionDefs().isEmpty();
        assert null == engine.getJsFunctionDef("no_this_function");
    }

    @Test
    @Order(2)
    void PutAndCallSimpleFunction() {
        assert engine != null;
        try {
            int res;
            engine.put("f01", 1, "function f01(a,b) { return a+b+1; }");

            for (int i = 0; i < 50; ++i) {
                long begin = System.currentTimeMillis();
                res = engine.call("f01", int.class, 1, 2);
                assert res == 4;
                long end = System.currentTimeMillis();
                System.out.println(end - begin);
            }
        } catch (Exception e) {
            e.printStackTrace();
            assert false;
        }
    }

    @Test
    @Order(3)
    void PutAndCallSimpleLambda() {
        assert engine != null;
        try {
            int res;
            engine.put("l01", 1, "(a,b) => { return a*b*2; }");

            for (int i = 0; i < 50; ++i) {
                long begin = System.currentTimeMillis();
                res = engine.call("l01", int.class, 2, 3);
                assert res == 12;
                long end = System.currentTimeMillis();
                System.out.println(end - begin);
            }
        } catch (Exception e) {
            e.printStackTrace();
            assert false;
        }
    }

    @Test
    @Order(4)
    void PutAndCallComplexFunction() {
        assert engine != null;
        try {
            Map<?,?> res;
            engine.put("f02", 1, "" +
                    "function f02(a,b) { " +
                    "   a.p1 = a.p1+'123'; " +
                    "   a.p2 = a.p2+123; " +
                    "   b.setP3('def'); " +
                    "   b.setP4(b.getP4()+1);" +
                    "   return {'a':a, 'b':b, 'c':'abc', 'd':4};" +
                    "}");

            for (int i = 0; i < 50; ++i) {
                long begin = System.currentTimeMillis();
                StructA a = new StructA("s1.1", 1, "s1.2", 2);
                StructA b = new StructA("s2.1", 8, "s2.2", 9);
                res = engine.call("f02", Map.class, a, b);
                assert res != null;
                assert res.size() == 4;
                assert "abc".equals(res.get("c"));
                assert res.get("d").equals(4);
                assert res.get("a").equals(a);
                assert res.get("b").equals(b);
                long end = System.currentTimeMillis();
                System.out.println(end - begin);
            }
        } catch (Exception e) {
            e.printStackTrace();
            assert false;
        }
    }

    @Test
    @Order(5)
    void PutAndCallOverrideLambda() {
        assert engine != null;
        try {
            int res;
            engine.put("l01", 2, "(a,b) => { return a*b*2+1; }");

            for (int i = 0; i < 50; ++i) {
                long begin = System.currentTimeMillis();
                res = engine.call("l01", int.class, 2, 3);
                assert res == 13;
                long end = System.currentTimeMillis();
                System.out.println(end - begin);
            }
        } catch (Exception e) {
            e.printStackTrace();
            assert false;
        }
    }

    @Test
    @Order(6)
    void RemoveFunction() {
        assert engine != null;
        try {
            int res;
            boolean onError;

            // call once before remove function.
            try {
                onError = false;
                res = engine.call("f01", int.class, 1, 2);
                assert res == 4;
            } catch (Exception ex) {
                onError = true;
            }
            assert !onError;

            // remove function
            engine.remove("f01");

            // call once after remove function.
            try {
                onError = false;
                res = engine.call("f01", int.class, 1, 2);
                assert res == 4;
            } catch (Exception ex) {
                onError = true;
            }
            assert onError;

        } catch (Exception e) {
            e.printStackTrace();
            assert false;
        }
    }
}
