package com.copote.integration.orchestration;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.camel.CamelContext;
import org.apache.camel.model.ProcessorDefinition;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

@Slf4j
@Component
public abstract class BaseParser implements ApplicationRunner {

    protected CamelContext _camelContext;

    @Autowired
    public final void set_camelContext(CamelContext value) {
        this._camelContext = value;
    }

    @Value("${integration.test.parser.print}")
    protected boolean PARSER_PRINTOUT;

    protected enum E_FieldType {
        Optional,
        Required,
    }

    protected enum E_EmptyType {
        NotEmpty,
        AllowEmpty,
    }

    protected static final String NAME_NODE = "node";

    protected static final String NAME_CHILDREN = "children";

    protected static final ThreadLocal<ValidateResult> s_ValidateResult;

    protected static final ConcurrentHashMap<String, BaseRootParser> s_RootParserMap;

    protected static final ConcurrentHashMap<String, BaseSubParser> s_SubParserMap;

    protected static final ConcurrentHashMap<String, BaseParser> s_ParserMap;

    static {
        s_ValidateResult = ThreadLocal.withInitial(ValidateResult::new);
        s_RootParserMap = new ConcurrentHashMap<>();
        s_SubParserMap = new ConcurrentHashMap<>();
        s_ParserMap = new ConcurrentHashMap<>();
    }

    protected abstract String nodeName();

    abstract protected boolean isRootNodeParser();

    public static @NonNull <T> T getParser(String name, Class<T> cls) throws RuntimeException {
        // check parser type
        if (cls != BaseRootParser.class && cls != BaseSubParser.class && cls != BaseParser.class)
            // should never reach
            throw new RuntimeException("coding error. parser type mismatch");

        // check name empty
        if (name == null || name.isEmpty())
            throw new RuntimeException("parser name is empty");

        // find parser with type conversion
        T parser;
        if (cls == BaseParser.class) parser = cls.cast(s_ParserMap.get(name));
        else if (cls == BaseSubParser.class) parser = cls.cast(s_SubParserMap.get(name));
        else parser = cls.cast(s_RootParserMap.get(name));

        // check result
        if (parser == null)
            throw new RuntimeException(String.format("parser %s not exist", name));
        return parser;
    }

    public static @NonNull <T> T getParser(@NonNull JSONObject json, Class<T> cls) throws RuntimeException {
        if (!json.containsKey(NAME_NODE))
            throw new RuntimeException("orchestration json node name not exist");
        return getParser(json.getString(NAME_NODE), cls);
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        try {
            if (null != s_ParserMap.putIfAbsent(nodeName(), this))
                throw new RuntimeException("parser coding error:" + this.nodeName());

            if (isRootNodeParser() && this instanceof BaseRootParser) {
                if (null != s_RootParserMap.putIfAbsent(nodeName(), (BaseRootParser) this))
                    throw new RuntimeException("parser coding error:" + this.nodeName());
            } else if ((!isRootNodeParser()) && this instanceof BaseSubParser) {
                if (null != s_SubParserMap.putIfAbsent(nodeName(), (BaseSubParser) this))
                    throw new RuntimeException("parser coding error:" + this.nodeName());
            } else {
                throw new RuntimeException("parser type conversion failed:" + this.nodeName());
            }
        } catch (Exception e) {
            // should never reach
            log.error(e.getMessage());
            throw e;
        }
    }

    public static ValidateResult Validate(@NonNull JSONObject json) {
        try {
            getParser(json.getString(NAME_NODE), BaseParser.class).base_validate(json);
            return s_ValidateResult.get().end();
        } catch (Exception e) {
            return s_ValidateResult.get().error(e).end();
        }
    }

    abstract protected void validate(@NonNull JSONObject json) throws Exception;

    protected final void base_validate(@NonNull JSONObject json) throws Exception {
        log.debug("validate node:" + nodeName());
        s_ValidateResult.get().pushPath(nodeName());
        validate(json);
        s_ValidateResult.get().popPath();
    }

    protected void ValidateNodeName(@NonNull JSONObject json) throws Exception {
        validateStringProperty(json, NAME_NODE, nodeName(), E_EmptyType.NotEmpty, E_FieldType.Required);
    }

    protected void ValidateChildren(@NonNull JSONObject json) throws Exception {
        validateArrayProperty(json, NAME_CHILDREN, E_FieldType.Optional);
    }

    protected void validateBoolProperty(@NonNull JSONObject json,
                                        @NonNull String key,
                                        E_FieldType required)
            throws Exception {
        Boolean input = json.getBoolean(key);

        if (required == E_FieldType.Optional && input == null)
            return;

        if (input == null)
            throw new JsonValidationException(nodeName(), key, "type");
    }

    protected void validateIntProperty(@NonNull JSONObject json,
                                        @NonNull String key,
                                        E_FieldType required)
            throws Exception {
        Integer input = json.getInteger(key);

        if (required == E_FieldType.Optional && input == null)
            return;

        if (input == null)
            throw new JsonValidationException(nodeName(), key, "type");
    }

    protected void validateDoubleProperty(@NonNull JSONObject json,
                                       @NonNull String key,
                                       E_FieldType required)
            throws Exception {
        Double input = json.getDouble(key);

        if (required == E_FieldType.Optional && input == null)
            return;

        if (input == null)
            throw new JsonValidationException(nodeName(), key, "type");
    }

    protected String validateStringProperty(@NonNull JSONObject json,
                                            @NonNull String key,
                                            E_EmptyType empty,
                                            E_FieldType required)
            throws Exception {
        String input = json.getString(key);

        if (required == E_FieldType.Optional && input == null)
            return null;

        if (input == null)
            throw new JsonValidationException(nodeName(), key, "type");

        if (empty == E_EmptyType.NotEmpty && json.getString(key).isEmpty())
            throw new JsonValidationException(nodeName(), key, "empty");

        return input;
    }

    protected void validateStringProperty(@NonNull JSONObject json,
                                          @NonNull String key,
                                          @NonNull String value,
                                          E_EmptyType empty,
                                          E_FieldType required)
            throws Exception {
        String input = validateStringProperty(json, key, empty, required);
        if (!value.equals(input) && required == E_FieldType.Required)
            throw new JsonValidationException(nodeName(), key, input);
    }

    protected void validateStringProperty(@NonNull JSONObject json,
                                          @NonNull String key,
                                          @NonNull Set<String> values,
                                          E_FieldType required)
            throws Exception {
        String input = validateStringProperty(json, key, E_EmptyType.AllowEmpty, required);
        if (!values.contains(input.toLowerCase()) && required == E_FieldType.Required)
            throw new JsonValidationException(nodeName(), key, input);
    }

    protected void validateArrayProperty(@NonNull JSONObject json,
                                         @NonNull String key,
                                         E_FieldType required)
            throws Exception {
        // validate node
        JSONArray array = json.getJSONArray(key);

        if (required == E_FieldType.Optional && array == null)
            return;

        if (array == null)
            throw new JsonValidationException(nodeName(), key, "type");

        // validate sub nodes
        JSONObject subJson;
        for (int i = 0; i < array.size(); ++i) {
            // push stack
            s_ValidateResult.get().pushPath(i);

            // check element type
            subJson = array.getJSONObject(i);
            if (subJson == null)
                throw new JsonValidationException(nodeName(), key, "type");

            // get parser and validate element
            getParser(subJson, BaseParser.class).base_validate(subJson);

            // pop stack
            s_ValidateResult.get().popPath();
        }
    }

    public static ValidateResult Parse(@NonNull JSONObject json) {
        try {
            getParser(json.getString(NAME_NODE), BaseRootParser.class).parse(json);
            return s_ValidateResult.get().end();
        } catch (Exception e) {
            return s_ValidateResult.get().error(e).end();
        }
    }

    protected ProcessorDefinition<?> parseChildren(@NonNull JSONArray children, ProcessorDefinition<?> def)
            throws Exception {

        // parse sub nodes
        JSONObject subJson;
        for (int i = 0; i < children.size(); ++i) {
            // check element type
            subJson = children.getJSONObject(i);
            if (subJson == null)
                throw new JsonValidationException(nodeName(), "children", "parse");

            // get parser and parse node
            def = getParser(subJson, BaseSubParser.class).parse(subJson, def);
        }
        return def;
    }
}
