package com.copote.integration.nodejs;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.pool2.ObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import java.util.concurrent.ConcurrentHashMap;

@Slf4j
@Scope("singleton")
@Component
public class NodeEngine {

    private final int _nodePoolMaxIdle;

    private final int _nodePoolMinIdle;

    private final int _nodePoolMaxTotal;

    private final ConcurrentHashMap<String, FunctionDefine> _jsFunctionDefs;

    private final ObjectPool<NodeContext> _nodeContextPool;

    public int get_nodePoolMaxIdle() {
        return _nodePoolMaxIdle;
    }

    public int get_nodePoolMinIdle() {
        return _nodePoolMinIdle;
    }

    public int get_nodePoolMaxTotal() {
        return _nodePoolMaxTotal;
    }

    public ConcurrentHashMap<String, FunctionDefine> get_jsFunctionDefs() {
        return _jsFunctionDefs;
    }

    public ObjectPool<NodeContext> get_nodeContextPool() {
        return _nodeContextPool;
    }

    public NodeEngine(@Value("${integration.nodejs.max_idle}") int nodePoolMaxIdle,
                      @Value("${integration.nodejs.min_idle}") int nodePoolMinIdle,
                      @Value("${integration.nodejs.max_total}") int nodePoolMaxTotal) {
        _nodePoolMaxIdle = nodePoolMaxIdle;
        _nodePoolMinIdle = nodePoolMinIdle;
        _nodePoolMaxTotal = nodePoolMaxTotal;

        _jsFunctionDefs = new ConcurrentHashMap<>();

        GenericObjectPoolConfig<NodeContext> config = new GenericObjectPoolConfig<>();
        config.setMaxIdle(_nodePoolMaxIdle);
        config.setMinIdle(_nodePoolMinIdle);
        config.setMaxTotal(_nodePoolMaxTotal);
        _nodeContextPool = new GenericObjectPool<>(new PooledNodeContextFactory(this), config);
    }

    public FunctionDefine getJsFunctionDef(String id) {
        return _jsFunctionDefs.get(id);
    }

    public void verify(String code) throws Exception {
        NodeContext context = null;
        try {
            context = _nodeContextPool.borrowObject();
            context.verify(code);
        } catch (Exception e) {
            log.debug("node engine verify code exception. " + e.getMessage());
            throw e;
        } finally {
            if (context != null)
                _nodeContextPool.returnObject(context);
        }
    }

    public void put(String id, long version, String code) throws Exception {
        verify(code);
        _jsFunctionDefs.put(id, FunctionDefine.Load(id, version, code));
    }

    public void remove(String id) throws Exception {
        _jsFunctionDefs.remove(id);
    }

    public <T> T call(String id, Class<T> type, Object... params) throws Exception {
        NodeContext context = null;
        try {
            context = _nodeContextPool.borrowObject();
            return context.call(id, type, params);
        } catch (Exception e) {
            log.debug("node engine execute code exception. " + e.getMessage());
            throw e;
        } finally {
            if (context != null)
                _nodeContextPool.returnObject(context);
        }
    }
}
