package com.copote.integration.nodejs;

import lombok.extern.slf4j.Slf4j;
import org.graalvm.polyglot.Context;
import org.graalvm.polyglot.HostAccess;
import org.graalvm.polyglot.Source;
import org.graalvm.polyglot.Value;

import java.io.Closeable;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.HashMap;

@Slf4j
public class NodeContext implements Closeable {
    private static class FunctionPtr {
        private final String _id;

        private final long _version;

        private final Value _function;

        FunctionPtr(String id, long version, Value function) {
            _id = id;
            _version = version;
            _function = function;
        }

    }

    private static final String S_permittedLanguages = "js";

    private static final String S_FunctionMapName = "iPaaSFuncMap";

    private final Context _context;

    private final HashMap<String, FunctionPtr> _functions;

    private final WeakReference<NodeEngine> _engine;

    public NodeContext(WeakReference<NodeEngine> engine) {
        _engine = engine;
        _functions = new HashMap<>();

        _context = Context.newBuilder(S_permittedLanguages)
                .allowCreateThread(true)
                .allowIO(true)
                .allowHostAccess(buildHostAccess())
                .allowAllAccess(true)
                .build();

        makeFunctionMap();
    }

    private HostAccess buildHostAccess() {
        return HostAccess.newBuilder(HostAccess.ALL)
                // number narrow
                .targetTypeMapping(Double.class, Float.class, null, Double::floatValue)
                .targetTypeMapping(Long.class, Integer.class, null, Long::intValue)

                // number to string
                .targetTypeMapping(Integer.class, String.class, null, Object::toString)
                .targetTypeMapping(Long.class, String.class, null, Object::toString)
                .targetTypeMapping(Short.class, String.class, null, Object::toString)
                .targetTypeMapping(Float.class, String.class, null, Object::toString)
                .targetTypeMapping(Double.class, String.class, null, Object::toString)

                // string to number
                .targetTypeMapping(String.class, Integer.class, null, Integer::parseInt)
                .targetTypeMapping(String.class, Long.class, null, Long::parseLong)
                .targetTypeMapping(String.class, Short.class, null, Short::parseShort)
                .targetTypeMapping(String.class, Float.class, null, Float::parseFloat)
                .targetTypeMapping(String.class, Double.class, null, Double::parseDouble)

                // to bool
                .targetTypeMapping(String.class, Boolean.class, null, Boolean::parseBoolean)
                .targetTypeMapping(Integer.class, Boolean.class, null, x -> x == 0)
                .targetTypeMapping(Long.class, Boolean.class, null, x -> x == 0)
                .targetTypeMapping(Short.class, Boolean.class, null, x -> x == 0)
                .targetTypeMapping(Float.class, Boolean.class, null, x -> x == 0)
                .targetTypeMapping(Double.class, Boolean.class, null, x -> x == 0)

                // from bool
                .targetTypeMapping(Boolean.class, String.class, null, Object::toString)
                .targetTypeMapping(Boolean.class, Integer.class, null, x -> x ? 1 : 0)
                .targetTypeMapping(Boolean.class, Long.class, null, x -> x ? 1L : 0L)
                .targetTypeMapping(Boolean.class, Short.class, null, x -> x ? (short) 1 : (short) 0)
                .targetTypeMapping(Boolean.class, Float.class, null, x -> x ? 1.F : 0.F)
                .targetTypeMapping(Boolean.class, Double.class, null, x -> x ? 1.0 : 0.0)

                // to void
                .targetTypeMapping(Object.class, Void.class, null, x -> null)

                // TODO: add more auto converter here

                .build();
    }

    @Override
    public void close() throws IOException {
        if (_context != null)
            _context.close();
    }

    private void makeFunctionMap() {
        _context.eval(S_permittedLanguages, "let " + S_FunctionMapName + " = [];");
    }

    public void verify(String code) throws Exception {
        try {
            Source source = Source.create(S_permittedLanguages, code);
            Value value = _context.parse(source);
        } catch (Exception ex) {
            log.debug("parse js code on exception. " + ex.getMessage());
            throw ex;
        }
    }

    private void remove(String id) {
        _functions.remove(id);
        _context.eval(S_permittedLanguages, String.format("delete %s['%s']",
                S_FunctionMapName,
                id));
    }

    private FunctionPtr load(FunctionDefine fd) {
        String code = String.format("%s['%s'] = (%s); %s['%s'];",
                S_FunctionMapName, fd.getID(), fd.getCode(),
                S_FunctionMapName, fd.getID());
        Value functionValue = _context.eval(S_permittedLanguages, code);
        FunctionPtr functionPtr = new FunctionPtr(fd.getID(), fd.getVersion(), functionValue);
        _functions.put(fd.getID(), functionPtr);
        return functionPtr;
    }

    public <T> T call(String id, Class<T> type, Object... params) throws Exception {
        // check engine
        NodeEngine engine = _engine.get();
        if (engine == null)
            throw new Exception("engine is stop");

        // check function define
        FunctionDefine fd = engine.getJsFunctionDef(id);
        if (fd == null) {
            remove(id);
            throw new Exception(String.format("function %s not exist", id));
        }

        // check function version
        FunctionPtr fp = _functions.get(id);
        if (fp == null || fp._version != fd.getVersion()) {
            // auto reload
            fp = load(fd);
        }

        // execute function
        Value rtn = fp._function.execute(params);
        return rtn.as(type);
    }
}
