package com.copote.integration.convert;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import java.io.FileInputStream;
import java.nio.charset.StandardCharsets;

public class ValidateJson {

    private static final String TEXT_RED = "\u001B[31m";
    private static final String TEXT_BLACK = "\u001B[30m";
    private static final String TEXT_GREEN = "\u001B[32m";
    private static final String TEXT_BLUE = "\u001B[34m";
    private static final String TEXT_RESET = "\u001B[0m";
    private static final String TEXT_PURPLE = "\u001B[35m";
    private static final String TEXT_CYAN = "\u001B[36m";
    private static final String TEXT_YELLOW = "\u001B[33m";
    private static final String TEXT_WHITE = "\u001B[37m";

    private enum Direction {
        SOURCE,
        TARGET,
    }

    private static boolean isNumber(Object value) {
        Class<?> type = value.getClass();
        return type == Integer.class || type == Double.class || type == Float.class || type == Long.class || type == Short.class;
    }

    private static boolean isBool(Object value) {
        Class<?> type = value.getClass();
        return type == Boolean.class;
    }

    private static boolean isString(Object value) {
        Class<?> type = value.getClass();
        return type == String.class;
    }

    private static boolean isObject(Object value) {
        Class<?> type = value.getClass();
        return type == JSONObject.class;
    }

    private static boolean isArray(Object value) {
        Class<?> type = value.getClass();
        return type == JSONArray.class;
    }

    public static void compare(JSONObject source, JSONObject target, int level) throws Exception {
        // start
        print(Direction.SOURCE, "{", source == null, level);
        if (target == null)
            print(Direction.TARGET, "__LOST__", true, level);
        if (source == null || target == null)
            // should never reach
            return;

        // element
        for (String key : source.keySet()) {
            // src
            Object sourceValue = source.get(key);
            printValue(key, source, sourceValue, level, Direction.SOURCE, false);

            // target
            Object targetValue = target.get(key);
            if (targetValue == null) {
                print(Direction.TARGET, "__LOST__", true, level + 1);
            } else if (targetValue.getClass() != sourceValue.getClass()) {
                print(Direction.TARGET, "__TYPE__", true, level + 1);
            } else {
                if (isNumber(targetValue) || isBool(targetValue) || isString(targetValue)) {
                    if (!targetValue.equals(sourceValue))
                        printValue(key, target, targetValue, level, Direction.TARGET, true);
                } else if (isObject(targetValue)) {
                    compare(source.getJSONObject(key), target.getJSONObject(key), level + 1);
                } else if (isArray(targetValue)) {
                    compare(source.getJSONArray(key), target.getJSONArray(key), level + 1);
                }
            }
        }

        // end
        print(Direction.SOURCE, "},", false, level);
    }

    private static void compare(JSONArray source, JSONArray target, int level) throws Exception {
        // start
        print(Direction.SOURCE, "[", source == null, level);
        if (target == null)
            print(Direction.TARGET, "__LOST__", true, level);
        if (source == null || target == null)
            // should never reach
            return;

        // children
        for (int i = 0; i < source.size(); ++i) {
            JSONObject sourceObject = source.getJSONObject(i);
            JSONObject targetObject = target.getJSONObject(i);
            compare(sourceObject, targetObject, level + 1);
        }

        // end
        print(Direction.SOURCE, "],", false, level);
    }

    private static void printValue(String key, JSONObject source, Object sourceValue, int level, Direction dir, boolean mismatch) throws Exception {
        if (isNumber(sourceValue)) {
            print(dir, String.format("'%s': %f,", key, source.getDoubleValue(key)), mismatch, level + 1);
        } else if (isBool(sourceValue)) {
            print(dir, String.format("'%s': %b,", key, source.getBooleanValue(key)), mismatch, level + 1);
        } else if (isString(sourceValue)) {
            print(dir, String.format("'%s': '%s',", key, source.getString(key)), mismatch, level + 1);
        } else if (isObject(sourceValue)) {
            print(dir, String.format("'%s': ", key), mismatch, level + 1);
        } else if (isArray(sourceValue)) {
            print(dir, String.format("'%s': ", key), mismatch, level + 1);
        }
    }

    private static void print(Direction dir, String text, boolean mismatch, int level) throws Exception {
        if (mismatch)
            System.out.print(TEXT_RED);

        if (dir == Direction.SOURCE)
            System.out.print("--> ");
        else
            System.out.print("--< ");

        printLevel(level);
        System.out.print(text);

        if (mismatch)
            System.out.print(TEXT_RESET);

        System.out.print(System.lineSeparator());

        if (mismatch)
            throw new Exception("error");
    }

    private static void printLevel(int level) {
        for (int i = 0; i < level * 2; ++i)
            System.out.print(' ');
    }

    public static void main(String[] args) throws Exception {
        JSONObject source = null;
        JSONObject target = null;
        try (FileInputStream fis = new FileInputStream(
                "/home/renjie/project/integration-core/test/convert2_tar.json")) {
            source = JSONObject.parseObject(fis, StandardCharsets.UTF_8, JSONObject.class);
        } catch (Exception ignore) {
        }
        try (FileInputStream fis = new FileInputStream(
                "/home/renjie/project/integration-core/test/convert2_tar_mis.json")) {
            target = JSONObject.parseObject(fis, StandardCharsets.UTF_8, JSONObject.class);
        } catch (Exception ignore) {
        }

        compare(source, target, 0);
    }
}
